#!/bin/bash

cdir=`pwd`
prog=`basename $0`
#-------------------------------------------------------------------------------------------
# Defaults:
  force=2
animate=1
#-------------------------------------------------------------------------------------------
# If you want newlines, you have to echo "$USAGE" and not echo $USAGE
USAGE="\
Usage: ${prog} -f <force> k1 k2 x10 x20 v10 v20 STEPS t0 tf
Other Options:
 -n Do not animate trajectory
Available forces (value of <force>):
1: ax= -k1 x          ay= -k2 y           Harmonic oscillator 
2: ax= 0              ay= -k1             Free fall           
3: ax= -k2     vx     ay= -k2    vy - k1  Free fall + air resitance ~ v
4: ax= -k2 |v| vx     ay= -k2 |v|vy - k1  Free fall + air resitance ~ v^2
5: ax= k1*x1/r^3      ay= k1*x2/r^3       Coulomb Force
6: ax= k1*x1/r^k2     ay= k1*x2/r^k2      power law central force
7: ax= f(r)*x1/r      ay= f(r)*x2/r       f(r)=1/r^2-r/k1^3, r<k1
8: ax= f(r)*x1/r      ay= f(r)*x2/r       f(r)=k1 e^(-r/k2) (1+r/k2) (Yukawa)"
#-------------------------------------------------------------------------------------------
set -u # Undo: set +u    Treat unset variables and parameters other than the special parameters "@" and "*" as an error 
while getopts :hnf: opt_char; do
 case $opt_char in
  n)  animate=0                                                 ;;
  f)  force=$OPTARG                                             ;;
  h)  echo "$USAGE"                                     ;exit 1 ;;
 \?)  echo "$OPTARG is not a valid option";echo "$USAGE";exit 1 ;;
 esac
done
shift $((OPTIND-1))

if(( $# != 9 ));then echo "$USAGE";exit;fi  # If arguments are necessary, uncomment this line

   k1=$1 
   k2=$2
  x10=$3
  x20=$4
  v10=$5
  v20=$6
STEPS=$7 ; (( STEPS++ )) # STEPS -> Nt
   t0=$8
   tf=$9

#          1   2 3  4   5  6   7  8
forcecode=(hoc g vg v2g cb pow hy yu)

if (( force > ${#forcecode[@]} ));then
 echo "${prog}: Not valid code for type of force"
 echo "$USAGE"
 exit 1
fi




if [[ -f rk4           ]];then rm rk4          ;fi
if [[ -f rk4_force.mod ]];then rm rk4_force.mod;fi

(( force-- ))
gfortran -O2 rk4_${forcecode[$force]}.f90 rk4.f90 -o rk4
./rk4 <<EOF
$k1 $k2
$STEPS $t0 $tf $x10 $x20 $v10 $v20
EOF


gnuplot -persist  -d <<EOF
set term qt 1
plot "rk4.dat" using 1:2 with lines title "x1(t)"
set term qt 2
plot "rk4.dat" using 1:3 with lines title "x2(t)"
set term qt 3
plot "rk4.dat" using 1:4 with lines title "v1(t)"
set term qt 4
plot "rk4.dat" using 1:5 with lines title "v2(t)"
set term qt 5
plot "rk4.dat" using 1:6 with lines title "E(t)"
set term qt 6
# set size square
set size ratio -1
set title "Trajectory"
plot "rk4.dat" using 2:3 with lines notit
EOF



if(( animate == 1));then
 skip=$(awk -v s=$STEPS 'BEGIN{s=int(s/100);if(s<1){s=1};print s}')
 ./rk4_animate.sh -d $skip -r
fi
#  ---------------------------------------------------------------------
#  Copyright by Konstantinos N. Anagnostopoulos (2004-2021)
#  Physics Dept., National Technical University,
#  konstant@mail.ntua.gr, www.physics.ntua.gr/~konstant
#  
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, version 3 of the License.
#  
#  This program is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#  
#  You should have received a copy of the GNU General Public Liense along
#  with this program.  If not, see <http://www.gnu.org/licenses/>.
#  -----------------------------------------------------------------------




